(*^
::[	Information =

	"This is a Mathematica Notebook file.  It contains ASCII text, and can be
	transferred by email, ftp, or other text-file transfer utility.  It should
	be read or edited using a copy of Mathematica or MathReader.  If you 
	received this as email, use your mail application or copy/paste to save 
	everything from the line containing (*^ down to the line containing ^*)
	into a plain text file.  On some systems you may have to give the file a 
	name ending with ".ma" to allow Mathematica to recognize it as a Notebook.
	The line below identifies what version of Mathematica created this file,
	but it can be opened using any other version as well.";

	FrontEndVersion = "Macintosh Mathematica Notebook Front End Version 2.2";

	MacintoshStandardFontEncoding; 
	
	fontset = title, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeTitle, center, M7, bold, e8,  24, "Times"; 
	fontset = subtitle, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeTitle, center, M7, bold, e6,  18, "Times"; 
	fontset = subsubtitle, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeTitle, center, M7, italic, e6,  14, "Times"; 
	fontset = section, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeSection, grayBox, M22, bold, a20,  18, "Times"; 
	fontset = subsection, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeSection, blackBox, M19, bold, a15,  14, "Times"; 
	fontset = subsubsection, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeSection, whiteBox, M18, bold, a12,  12, "Times"; 
	fontset = text, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = smalltext, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  10, "Times"; 
	fontset = input, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeInput, M42, N23, bold, L-5,  10, "Courier"; 
	fontset = output, output, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeOutput, M42, N23, L-5,  10, "Courier"; 
	fontset = message, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeOutput, M42, N23, R65535, L-5,  12, "Courier"; 
	fontset = print, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeOutput, M42, N23, L-5,  10, "Courier"; 
	fontset = info, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeOutput, M42, N23, B65535, L-5,  12, "Courier"; 
	fontset = postscript, PostScript, formatAsPostScript, output, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeGraphics, M7, l34, w245, h249,  12, "Courier"; 
	fontset = name, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7, italic,  10, "Geneva"; 
	fontset = header, inactive, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = leftheader, inactive, L2,  12, "Times"; 
	fontset = footer, inactive, noKeepOnOnePage, preserveAspect, center, M7,  12, "Times"; 
	fontset = leftfooter, inactive, L2,  12, "Times"; 
	fontset = help, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  10, "Times"; 
	fontset = clipboard, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = completions, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special1, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special2, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special3, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special4, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special5, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	currentKernel; 
]
:[font = title; inactive; locked; preserveAspect; startGroup]
Simulating Bugs in Feynman's Quantum Computer
:[font = subsubtitle; inactive; locked; preserveAspect]
Colin P. Williams & Scott H. Clearwater
:[font = input; initialization; preserveAspect]
*)
Needs["Statistics`ContinuousDistributions`"]; 
Needs["Statistics`DiscreteDistributions`"]; 
Needs["Statistics`DataManipulation`"]; 
Needs["Graphics`Graphics`"]; 
Off[General::spell]; 
Off[General::spell1]; 
(*
:[font = section; inactive; locked; preserveAspect; startGroup]
Copyright Notice
:[font = text; inactive; locked; preserveAspect; endGroup]
Copyright Colin P. Williams & Scott H. Clearwater (1997).

This Notebook is intended to be used in conjunction with "Explorations in Quantum Computing" by Colin P. Williams and Scott H. Clearwater, TELOS, Springer-Verlag (1997), ISBN:0-387-94768-X. Permission is hereby granted to copy and distribute this Notebook freely for any non-commercial activity provided you include this copyright notice at the beginning of all such copies. Please send suggestions and bug reports to Colin P. Williams at 
        colin@solstice.jpl.nasa.gov      (818) 306 6512 or 
        cpw@cs.stanford.edu               (415) 728 2118
For information on "Explorations in Quantum Computing" check out the TELOS web site:  http://www.telospub.com/catalog/PHYSICS/Explorations.html. To order call 1-800-777-4643.

All other rights reserved.
:[font = section; inactive; locked; preserveAspect; startGroup]
Introduction
:[font = text; inactive; locked; preserveAspect; endGroup]
This Notebook contains code that allows you to experiment with the effects of imperfections on the performance of Feynman's quantum computer (see "Explorations in Quantum Computing" Chapter 5).  An imperfection, such as an atom being slightly misaligned, will cause the actual Hamiltonian of the quantum computer, Hbuggy, to differ slightly from the intended Hamiltonian, H. This in turn will induce an error in the unitary evolution operator, U, through the relationship U=Exp[I H t/hBar] (where hBar is Planck's constant divided by 2 Pi). The unitary evolution operator is really the "program" that the quantum computer is following. Thus if the form of the actual unitary evolution operator is Ubuggy = Exp[I Hbuggy t/hBar] rather than U, this means that the quantum computer will perform a computation other than the one intended. However, if the errors in the Hamiltonian are not too great then it is possible that the quantum computer will still perform the correct computation accidentally. The question is how bad can the errors be before the output from the quantum computer is essentially useless? The code in this Notebook helps to answer this question.
:[font = section; inactive; locked; preserveAspect; startGroup]
>>> README >>>
:[font = text; inactive; locked; preserveAspect; endGroup]
If you are using Mathematica version 2.2 you must open and initialize all input cells in the Notebook Feynman.ma before the code in this Notebook (QBugs.ma) will work. If you are using Mathematica v3.0 use Feynman.nb and QBugs.nb instead. Both Feynman.ma(.nb) and QBugs.ma(.nb) must be open and initialized before the code for simulating bugs in Feynman's quantum computer will work properly.
:[font = section; inactive; preserveAspect; startGroup]
Circuit for NOT Built From Two SqrtNOT Gates
:[font = text; inactive; preserveAspect]
We are going to use our good old sqrtNOTcircuit that was introduced in Chapter 4. This circuit computes the NOT function as two SqrtNOT gates connected back to back. The code that defines the circuit is contained in the Feynman.ma(.nb) Notebook. The circuit is represented as a pair of square root of NOT gates.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = text; inactive; preserveAspect]
Here is the definition of the sqrtNOTcircuit. To see the circuit representation click on "sqrtNOTcircuit" in the cell below and hit the RETURN key whilst holding down the SHIFT key simultaneously:
:[font = input; preserveAspect]
sqrtNOTcircuit
:[font = text; inactive; preserveAspect; endGroup; endGroup]
If you do not see a list of two matrices when you evaluated sqrtNOTcircuit it means that you have not initialized the Feynman.ma(.nb) Notebook. In version 3.0 you initialize a Notebook by using the Kernel menu and selecting the Kernel/Evaluation/Evaluate Initialization sub-menu. In version 2.2 you initialize a Notebook using the Action menu by selecting the Action/Evaluate Initialization sub-menu.
:[font = section; inactive; preserveAspect; startGroup]
Correct Behavior of the Quantum Computer
:[font = text; inactive; preserveAspect; endGroup]
If the computation to be performed is NOT computed as two SqrtNOT gates connected back to back then we have SqrtNOT(SqrtNOT(x)) = x. If the input bit is x, the input state (which includes the auxiliary cursor bits) is the state ket[1,0,0,x]: the first three qubits represent the cursor and the last qubit, x, represents the input bit. After the quantum computer has evolved the memory register ought to evolve into the state ket[0,0,1,NOT(x)]. Note that the cursor advances from left to right in such a way that there is only ever one cursor bit set to 1. Thus if the input state is ket[1,0,0,0] the output should be ket[0,0,1,1]. Conversely, if the input state is ket[1,0,0,1] the output state ought to be ket[0,0,1,0]. The sqrtNOTcircuit can handle superpositions of inputs but we will restrict attention to just classical inputs throughout this Notebook. The reason we do this is because you cannot assess the correctness of a quantum computation on a superposed input state in a single run. To test that would need to do several runs and compare the distribution of "answers" (note plural) that  you see with the distribution of answers that you expect. So for our purposes we will use only classical inputs and expect only classical outputs.

We regard the computation as correct if both the answer bit is "correct" and the cursor is "uncorrupted". By a "correct" answer bit we mean, if the input to the computer was the binary digit x then, when the cursor bit was found to be in its terminal (i.e. third) position, the output memory register bit was NOT(x). By an "uncorrupted" cursor we mean at those prior times when the cursor was measured and found not to be at its terminal position, there was at most one 1 amongst the three cursor bits. Thus by restricting attention to classical binary inputs (0 or 1 and not superpositions of 0 and 1) we can determine whether the computation is correct simply by reading the cursor bits (at every time step) and the answer bit (at the last time step when the cursor is at its terminal position).
:[font = section; inactive; preserveAspect; startGroup]
Modeling Errors in the Hamiltonian
:[font = text; inactive; preserveAspect]
The Hamiltonian of a quantum computer is described mathematically by a hermitian matrix. This is a matrix whose (i,j)-th element is the complex conjugate of its (j,i)-th element. Even the "buggy" Hamiltonian representing a faulty computation must be described by some hermitian matrix. So whetever we do to create a buggy Hamiltonian from the correct Hamiltonian, we had better be sure that the operation preserves hermiticity. 

To model errors in the Hamiltonian, we start by computing the correct Hamiltonian for the desired circuit (i.e. computation) using the function Hamiltonian defined in the Feynman.ma(.nb) Notebook, and then adding random normally distributed noise to the upper triangular elements of the Hamiltonian. We then set the lower triangular elements to be the complex conjugates of the "buggy" upper triangular elements and voila! we have a buggy Hamiltonian. By assumption we set the mean error to be 0 and the standard deviation in the error to be stdDevError. Thus stdDevError is the parameter that determines how far we nudge the elements of the Hamiltonian away from their indented values. A reasonable value to use for stdDevError (if you want to mimic the effects of small to medium errors) is to have 0 <= stdDevError <= 1. 

This procedure gives us a hermitian matrix (buggy Hamiltonian) that is "close" to the correct hermitian matrix (correct Hamiltonian). However, there are certainly other more physically motivated ways to generate a buggy Hamiltonian. Nevertheless, our simple mathematical is adequate to illustrate how errors grow due to imperfections.
:[font = input; initialization; preserveAspect]
*)
BuggyHamiltonian[m_, k_, circuit_, stdDevError_] := 
  Module[{ham}, ham = Hamiltonian[m, k, circuit]; 
    For[i = 1, i <= 2^(m + k + 1), i++, 
     For[j = i, j <= 2^(m + k + 1), j++, 
       ham[[i,j]] = 
         ham[[i,j]] + Abs[Random[NormalDistribution[0., stdDevError]]] + 
          I*Random[NormalDistribution[0., stdDevError]]; 
        ham[[j,i]] = Conjugate[ham[[i,j]]]; ]; ]; ham]
(*
:[font = text; inactive; preserveAspect; endGroup]
A buggy Hamiltonian automatically induces a buggy unitary evolution operator. So we will overwrite the old definition of EvolutionOP in our Feynman.nb simulator. The new definition of EvolutionOP uses the buggy Hamiltonian.
:[font = section; inactive; preserveAspect; startGroup]
Simulating Bugs in Feyman's Quantum Computer
:[font = text; inactive; preserveAspect]
We are now ready to try to simulate bugs in Feynman's quantum computer. The top level function is SimulateBugsInFeynmanQC. This function takes 6 arguments:

(1) The initial state fed into the quantum computer.
(2) The correct output state.
(3) The quantum circuit (i.e. the computation to be performed)
(4) The number of runs of the simulator using this input and this circuit.
(5) The standard deviation of the random "noise" terms added to the Hamiltonian matrix elements
    (typically between 0 and 1)
(6) The name of a file in which to append the output data.
:[font = text; inactive; preserveAspect]
The output is a 10-argument list:
(A) The standard deviation of the errors that was used (this was input 5 above)
(B) Number of steps to termination for those runs that yielded a correct computation (right answer/good cursor)
(C) The mean or (B)
(D) The standard deviation of (B)
(E) Number of steps to termination for those runs that yielded a faulty computation (wrong answer/good cursor)
(F) The mean of (E)
(G) The standard deviation of (E)
(H) Number of steps to termination for those runs that yielded an unreliable computation (bad cursor)
(I) The mean of (H)
(J) The standard deviation of (H)
:[font = text; inactive; preserveAspect]
If you forget what the inputs are ask for the usage of SimulateBugsInFeynmanQC.
:[font = input; preserveAspect; startGroup]
?SimulateBugsInFeynmanQC
:[font = print; inactive; preserveAspect; endGroup]
SimulateBugsInFeynmanQC[initState,correctAns,circuit,n,stdDevError,file] performs n
   independent runs of a Feynman quantum computer for the given circuit starting from
   the given initial state. The Hamiltonian for this circuit is "corrupted" by
   introducing random normally distributed errors to its matrix elements. The errors
   have a mean of zero and a standard deviation of stdDevError. The errors are
   introduced in such a way as to keep the "buggy" Hamiltonian hermitian. The output is
   a 10 element list that shows the stdDevError, the number of steps to termination of
   runs that ended with the correct answer (and legal cursor), their mean and standard
   deviation, the number of steps to termination for runs that ended with the wrong
   answer (but legal cursor), their mean and standard deviation and the number of steps
   to termination for runs that ended with an illegal cursor (i.e. a cursor with more
   than one 1) whether the answer was correct or incorrect and their mean and standard
   deviation. The output is printed to the screen and recorded in the given file.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try Me!
:[font = text; inactive; preserveAspect]
Let's begin by seeing what happens when we ask for 25 runs of a Feynman quantum computer used to compute the NOT of 1 when the standard deviation in the noise added to the Hamiltonian is 0.05. Remember you must open and evaluate the initialization cells in Feynman.ma(.nb) in order for SimulateBugsInFeynmanQC code to work.

The input state contains the initial specification of the cursor, {1,0,0}, together with the input to the circuit, the binary value 1. We enter the 4 bits {1,0,0,1} as the 4-particle quantum state ket[1,0,0,1].  The computation is supposed to be "complete and correct" whenever the cursor is observed to be in its terminal position (the 1 appears on the (k+1)-th qubit. Thus if we measure the state of just the cursor qubits (the leftmost k+1=3 qubits in our current case) and find them in the state ket[0,0,1,?x] where ?x is unmeasured, we know that at that moment a measurement of ?x ought to reveal the correct answer to the computation, i.e. ?x ought to be 0. Therefore, if the input state is ket[1,0,0,1] and the computer is performing the NOT computation correctly, the entire memory register should evolve into the state ket[0,0,1,0] . Try the following example and see what you get.
:[font = input; preserveAspect; startGroup]
SimulateBugsInFeynmanQC[ket[1, 0, 0, 1], ket[0, 0, 1, 0], sqrtNOTcircuit, 25, 
  0.05, "test.dat"]
:[font = print; inactive; preserveAspect]
Simulating errors at the level stdDevError = 0.05
Output data is appended to the file test.dat
Number of gates in circuit, k = 2
Number of qubits in cursor, k+1 = 3
Number of qubits for input to/output from circuit, m = 1
Total number of qubits in memory register m+k+1 = 4
Number of runs = 25
Completed 12 runs out of 25
	Fraction of runs that are corrupted i.e. have bad cursor = 0.0833333
Completed 24 runs out of 25
	Fraction of runs that are corrupted i.e. have bad cursor = 0.166667
CPU time for simulation = 456.00000 seconds



Fraction of runs that were successful
	(i.e. gave correct answer and had legal cursor) = 0.72
Average time to complete a successful run = 3.11111 +/- 2.69834

Fraction of runs that were unsuccessful
	(i.e. gave wrong answer but had legal cursor) = 0.12
Average time to complete an unsuccessful run = 5.66667 +/- 3.51188

Fraction of runs that were inconclusive
	because they had a bad cursor = 0.16
Average time to detect bad cursor = 1.5 +/- 1.
:[font = output; output; inactive; preserveAspect; endGroup; endGroup]
{0.05, {1, 3, 1, 4, 3, 1, 6, 12, 2, 1, 2, 2, 6, 3, 2, 1, 3, 3}, 
 
  3.111111111111111, 2.698341296358557, {6, 2, 9}, 5.666666666666666, 
 
  3.511884584284246, {3, 1, 1, 1}, 1.5, 1.}
;[o]
{0.05, {1, 3, 1, 4, 3, 1, 6, 12, 2, 1, 2, 2, 6, 3, 2, 1, 3, 3}, 3.111111111111111, 2.698341296358557, {6, 2, 9}, 5.666666666666666, 3.511884584284246, {3, 1, 1, 1}, 1.5, 1.}
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = text; inactive; preserveAspect]
If you increase the magnitude of the errors in the Hamiltonian the computation becomes more unreliable. The following example illustrates what happens when the standard deviation of the error in the Hamiltonian is 0.5. Note that if we use the same filename as above, "test.dat", the new answers (with stdDevError = 0.5) will be appended to the end of the file.
:[font = input; preserveAspect; startGroup]
SimulateBugsInFeynmanQC[ket[1, 0, 0, 1], ket[0, 0, 1, 0], sqrtNOTcircuit, 25, 
  0.5, "test.dat"]
:[font = print; inactive; preserveAspect]
Simulating errors at the level stdDevError = 0.5
Output data is appended to the file test.dat
Number of gates in circuit, k = 2
Number of qubits in cursor, k+1 = 3
Number of qubits for input to/output from circuit, m = 1
Total number of qubits in memory register m+k+1 = 4
Number of runs = 25
Completed 12 runs out of 25
	Fraction of runs that are corrupted i.e. have bad cursor = 0.666667
Completed 24 runs out of 25
	Fraction of runs that are corrupted i.e. have bad cursor = 0.708333
CPU time for simulation = 179.00000 seconds



Fraction of runs that were successful
	(i.e. gave correct answer and had legal cursor) = 0.16
Average time to complete a successful run = 1. +/- 0

Fraction of runs that were unsuccessful
	(i.e. gave wrong answer but had legal cursor) = 0.12
Average time to complete an unsuccessful run = 1.33333 +/- 0.57735

Fraction of runs that were inconclusive
	because they had a bad cursor = 0.72
Average time to detect bad cursor = 1.11111 +/- 0.323381
:[font = output; output; inactive; preserveAspect; endGroup; endGroup]
{0.5, {1, 1, 1, 1}, 1., 0, {2, 1, 1}, 1.333333333333333, 0.5773502691896258, 
 
  {1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 2, 1, 1, 1, 1, 1}, 1.11111111111111, 
 
  0.3233808333817771}
;[o]
{0.5, {1, 1, 1, 1}, 1., 0, {2, 1, 1}, 1.333333333333333, 0.5773502691896259, {1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 2, 1, 1, 1, 1, 1}, 1.11111111111111, 0.3233808333817771}
:[font = subsubsection; inactive; preserveAspect; startGroup]
Code for SimulateBugsInFeynmanQC in here ...
:[font = input; initialization; preserveAspect]
*)
Options[SimulateBugsInFeynmanQC] = {Report -> True}; 
SimulateBugsInFeynmanQC[(w_.)*ket[bits__] + (kets_.), correctAns_, circuit_, 
   nRuns_, stdDevError_, filename_, opts___] := 
  Module[{reportQ, m, k, x, starttime, ham, evoln, timesRightAnsGoodCur, 
    timesWrongAnsGoodCur, timesNoAnsBadCur, runs, stream, meanR, meanW, 
    meanB, stdDevR, stdDevW, stdDevB}, 
   reportQ = Report /. {opts} /. Options[SimulateBugsInFeynmanQC]; 
    Off[EvolveQC::badCursor]; Clear[$hamiltonian]; 
    Print["Simulating errors at the level stdDevError = ", stdDevError]; 
    If[reportQ, Print["Output data is appended to the file ", filename]]; 
    k = Length[circuit]; If[reportQ, 
     Print["Number of gates in circuit, k = ", k]; 
      Print["Number of qubits in cursor, k+1 = ", k + 1]]; 
    m = Length[{bits}] - (k + 1); 
    If[reportQ, Print["Number of qubits for input to/output from circuit, m \
= ", m]; Print["Total number of qubits in memory register m+k+1 = ", 
       m + k + 1]]; x = {}; If[reportQ, Print["Number of runs = ", nRuns]]; 
    starttime = AbsoluteTime[]; 
    For[runs = 1, runs <= nRuns, runs++, 
     $hamiltonian = BuggyHamiltonian[m, k, circuit, stdDevError]; 
      (Clear[EvolutionOP]; EvolutionOP[_, _, _, t_] := 
         MatrixPower[N[MatrixExp[-(I*$hamiltonian)]], t]); 
      evoln = EvolveQC[w*ket[bits] + kets, circuit]; 
      x = Append[x, Last[evoln]]; 
      If[Mod[runs, If[nRuns < 10, 2, Round[nRuns/2]]] == 0, 
       Print["Completed ", runs, " runs out of ", nRuns]; 
        Print["\tFraction of runs that are corrupted i.e. have bad cursor = \
", N[Length[Select[x, #1[[3]] == $BadCursor & ]]/Length[x]]]; ]; ]; 
    Print["CPU time for simulation = ", AbsoluteTime[] - starttime, 
     " seconds", "\n"]; {timesRightAnsGoodCur, meanR, stdDevR, 
      timesWrongAnsGoodCur, meanW, stdDevW, timesNoAnsBadCur, meanB, 
      stdDevB} = ReadAnswerAndSummarizeRuns[x, correctAns, Report -> reportQ]\
; stream = OpenAppend[filename]; 
    Write[stream, {stdDevError, timesRightAnsGoodCur, meanR, stdDevR, 
      timesWrongAnsGoodCur, meanW, stdDevW, timesNoAnsBadCur, meanB, stdDevB}
]; Close[stream]; {stdDevError, timesRightAnsGoodCur, meanR, stdDevR, 
     timesWrongAnsGoodCur, meanW, stdDevW, timesNoAnsBadCur, meanB, stdDevB}]

SimulateBugsInFeynmanQC::usage = "SimulateBugsInFeynmanQC[initState,correctAn\
s,circuit,n,stdDevError,file] performs n independent runs of a Feynman \
quantum computer for the given circuit starting from the given initial state. \
The Hamiltonian for this circuit is \"corrupted\" by introducing random \
normally distributed errors to its matrix elements. The errors have a mean of \
zero and a standard deviation of stdDevError. The errors are introduced in \
such a way as to keep the \"buggy\" Hamiltonian hermitian. The output is a 10 \
element list that shows the stdDevError, the number of steps to termination \
of runs that ended with the correct answer (and legal cursor), their mean and \
standard deviation, the number of steps to termination for runs that ended \
with the wrong answer (but legal cursor), their mean and standard deviation \
and the number of steps to termination for runs that ended with an illegal \
cursor (i.e. a cursor with more than one 1) whether the answer was correct or \
incorrect and their mean and standard deviation. The output is printed to the \
screen and recorded in the given file."; 
(*
:[font = text; inactive; preserveAspect]
ReadAnswerAndSummarizeRuns analyzes the final descriptions of n runs and creates summary statistics which can be appended to a file of your choice. We regard the answer as correct if (a) the cursor bits contain at most one bit set to "1" and (b) the answer in the memory register is indeed the correct answer to the computation. Note that in order to be able to assess the correctness of the answer, we restrict ourselves to running simulations of computations that are function evaluations over classical inputs.  Our Feynman quantum computer will work on superpositions of inputs too but the analysis of the errors becomes more complicated. Note however, in the case of classical inputs were we expect only one legitimate answer we can compare the predicted output state to the actual output state. The computation is deemed correct and reliable if they are the same up to an arbitrary phase factor i.e. the following test returns True.
:[font = input; preserveAspect; startGroup]
MatchQ[actualOutputState, _. predictedOutputState]
:[font = output; output; inactive; preserveAspect; endGroup]
True
;[o]
True
:[font = text; inactive; preserveAspect; endGroup]
ReadAnswerAndSummarizeRuns reads a single output from each independent run of the quantum computer and records statistics on the the number of times the computation ended with the correct answer, the incorrect answer or an unreliable answer. Each run of the quantum computer is supposedly doing the "same" computation with different but equally buggy Hamiltonians. Note that ReadAnswerAnsSummarizeRuns is non-deterministic: two identical calls to ReadAnswerAndSummarizeRuns can result in different outputs due to the essential nondeterminsin inherent in the quantum mechanics of measurement on states that are superpositions of possible answers.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Code for ReadAnswerAndSummarizeRuns in here ...
:[font = input; initialization; preserveAspect]
*)
Options[ReadAnswerAndSummarizeRuns] = {Report -> False}; 

ReadAnswerAndSummarizeRuns[terminalDescriptions_, correctAns_, opts___] := 
  Module[{reportQ, pairs, tRAnsGCur, tWAnsGCur, tBadCur, mr, sr, mw, sw, mb, 
    sb}, reportQ = Report /. {opts} /. Options[ReadAnswerAndSummarizeRuns]; 
    pairs = ({#1, ReadMemoryRegister[ExtractFinalState[#1]][[2]]} & ) /@ 
      terminalDescriptions; tRAnsGCur = 
     (ExtractTime[#1[[1]]] & ) /@ RightAnsAndGoodCursor[pairs, correctAns]; 
    tWAnsGCur = 
     (ExtractTime[#1[[1]]] & ) /@ WrongAnsAndGoodCursor[pairs, correctAns]; 
    tBadCur = (ExtractTime[#1[[1]]] & ) /@ UnreliableAnsAndBadCursor[pairs]; 
    mr = N[myMean[tRAnsGCur]]; sr = N[myStdDev[tRAnsGCur]]; 
    mw = N[myMean[tWAnsGCur]]; sw = N[myStdDev[tWAnsGCur]]; 
    mb = N[myMean[tBadCur]]; sb = N[myStdDev[tBadCur]]; 
    If[reportQ === True, ReportFractionRightAnsGoodCursor[{tRAnsGCur, 
        tWAnsGCur, tBadCur}, mr, sr]; 
      ReportFractionWrongAnsGoodCursor[{tRAnsGCur, tWAnsGCur, tBadCur}, mw, 
       sw]; ReportFractionBadCursor[{tRAnsGCur, tWAnsGCur, tBadCur}, mb, sb]]\
; {tRAnsGCur, mr, sr, tWAnsGCur, mw, sw, tBadCur, mb, sb}]

RightAnsAndGoodCursor[pairs_, correctAns_] := 
  Select[Select[pairs, MatchQ[#1[[2]], _. correctAns] & ], 
   GoodCursorQ[ExtractCursor[#1[[1]]]] & ]

WrongAnsAndGoodCursor[pairs_, correctAns_] := 
  Select[Select[pairs,  !MatchQ[#1[[2]], _. correctAns] & ], 
   GoodCursorQ[ExtractCursor[#1[[1]]]] & ]

UnreliableAnsAndBadCursor[pairs_] := 
  Select[pairs, ExtractCursor[#1[[1]]] === $BadCursor & ]

GoodCursorQ[$BadCursor] := False
GoodCursorQ[cursorBits_List] := 
	If[Count[cursorBits, 1] == 1, True, False]

ExtractTime[{time_, _, _, _}] := time
ExtractCursor[{_, _, cursor_, _}] := cursor
ExtractFinalState[{_, _, _, finalState_}] := finalState
(*
:[font = input; initialization; preserveAspect]
*)
ReportFractionRightAnsGoodCursor[{r_, w_, b_}, mean_, stdDev_] := 
  (Print["\n"]; Print["Fraction of runs that were successful"]; 
    Print["\t(i.e. gave correct answer and had legal cursor) = ", 
     N[Length[r]/Length[Flatten[{r, w, b}]]]]; 
    Print["Average time to complete a successful run = ", mean, " +/- ", 
     stdDev, "\n"])

ReportFractionWrongAnsGoodCursor[{r_, w_, b_}, mean_, stdDev_] := 
  (Print["Fraction of runs that were unsuccessful"]; 
    Print["\t(i.e. gave wrong answer but had legal cursor) = ", 
     N[Length[w]/Length[Flatten[{r, w, b}]]]]; 
    Print["Average time to complete an unsuccessful run = ", mean, " +/- ", 
     stdDev, "\n"])

ReportFractionBadCursor[{r_, w_, b_}, mean_, stdDev_] := 
  (Print["Fraction of runs that were inconclusive"]; 
    Print["\tbecause they had a bad cursor = ", 
     N[Length[b]/Length[Flatten[{r, w, b}]]]]; 
    Print["Average time to detect bad cursor = ", mean, " +/- ", stdDev])
(*
:[font = input; initialization; preserveAspect; endGroup; endGroup]
*)
myMean[{}] := 0
myMean[x_] := Mean[x]
myStdDev[{}] := 0
myStdDev[{_}] := 0
myStdDev[x_] := StandardDeviation[x]
(*
:[font = section; inactive; preserveAspect; startGroup]
Creating and Visualizing Buggy Quantum Computations
:[font = text; inactive; preserveAspect]
We now have all the machinery necessary to simulate the effects of errors in the Hamiltonian of a quantum computer on the correctness and reliability of the answers obtained from the computer. The function CollectBugStatistics performs several runs of a quantum computer engaged in a specific computation as a function of the level of the error in the Hamiltonian (stdDevError). You can specify stdDevError to be in the range stdDevErrorMin <= stdDevError <= stdDevErrorMax, and to sweep through that range in steps of size stdDevErrorIncr.
:[font = input; preserveAspect; startGroup]
?CollectBugStatistics
:[font = print; inactive; preserveAspect; endGroup]
CollectBugStatistics[inputState, correctOutputState, circuit, n, {stdDevErrorMin,
   stdDevErrorMax, stdDevErrorIncr},file] performs n runs of the computation implied by
   the given circuit for each value of the error in the range stdDevErrorMin to
   stdDevErrorMax in steps of size stdDevErrorIncr. The output is the set of summary
   statistics for each data point as returned by a call to SimulateBugsInFeynmanQC for
   a specific value of stdDevError.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = text; inactive; preserveAspect]
Here we explore runs of the NOT computation implemented as two square root of NOT gates connected back to back. The input state is ket[1,0,0,1] (i.e. a starting cursor 100 and an input bit 1). The correct output state is ket[0,0,1,0] (i.e. an ending cursor 001 and an output bit NOT(1) = 0). We sweep out values of the error in the Hamiltonian ranging from 0 to 0.2 in intervals 0.03. There are 15 samples per data point. This number is too low for serious work but illustrates how to call CollectBugStatistics. all outputs are written to the file logfile.dat.
:[font = input; preserveAspect; startGroup]
bugStatistics = 
  CollectBugStatistics[ket[1, 0, 0, 1], ket[0, 0, 1, 0], sqrtNOTcircuit, 15, 
   {0, 0.2, 0.03}, "logfile.dat"]
:[font = print; inactive; preserveAspect]
Simulating errors at the level stdDevError = 0
Completed 8 runs out of 15
	Fraction of runs that are corrupted i.e. have bad cursor = 0
CPU time for simulation = 207.00000 seconds

Simulating errors at the level stdDevError = 0.03
Completed 8 runs out of 15
	Fraction of runs that are corrupted i.e. have bad cursor = 0
CPU time for simulation = 181.00000 seconds

Simulating errors at the level stdDevError = 0.06
Completed 8 runs out of 15
	Fraction of runs that are corrupted i.e. have bad cursor = 0.125
CPU time for simulation = 202.00000 seconds

Simulating errors at the level stdDevError = 0.09
Completed 8 runs out of 15
	Fraction of runs that are corrupted i.e. have bad cursor = 0.25
CPU time for simulation = 200.00000 seconds

Simulating errors at the level stdDevError = 0.12
Completed 8 runs out of 15
	Fraction of runs that are corrupted i.e. have bad cursor = 0.625
CPU time for simulation = 165.00000 seconds

Simulating errors at the level stdDevError = 0.15
Completed 8 runs out of 15
	Fraction of runs that are corrupted i.e. have bad cursor = 0.125
CPU time for simulation = 178.00000 seconds

Simulating errors at the level stdDevError = 0.18
Completed 8 runs out of 15
	Fraction of runs that are corrupted i.e. have bad cursor = 0.5
CPU time for simulation = 128.00000 seconds

:[font = output; output; inactive; preserveAspect; endGroup; endGroup]
{{0, {11, 7, 1, 2, 1, 2, 1, 4, 2, 9, 5, 2, 12, 3, 2}, 4.266666666666665, 
 
   3.731462116089284, {}, 0, 0, {}, 0, 0}, 
 
  {0.03, {2, 3, 2, 4, 2, 5, 2, 1, 2, 1, 1, 5, 1}, 2.384615384615384, 
 
   1.445594545418454, {}, 0, 0, {1, 1}, 1., 0}, 
 
  {0.06, {1, 5, 1, 1, 1, 6, 3, 1, 1, 2, 6}, 2.545454545454544, 
 
   2.114882330704777, {3}, 3., 0, {3, 2, 4}, 3., 1.}, 
 
  {0.09, {4, 2, 2, 1, 2, 3, 7, 2, 1, 5, 5}, 3.09090909090909, 
 
   1.921173883569388, {2}, 2., 0, {1, 1, 1}, 1., 0}, 
 
  {0.1199999999999999, {1, 2, 2, 1, 1, 1, 5, 7}, 2.5, 2.267786838055362, {3}, 
 
   3., 0, {2, 2, 1, 1, 1, 1}, 1.333333333333333, 0.516397779494322}, 
 
  {0.15, {1, 1, 1, 4, 1, 2, 1}, 1.57142857142857, 1.133893419027681, 
 
   {10, 4, 1}, 5., 4.582575694955838, {1, 1, 3, 1, 2}, 1.6, 0.894427190999916}
 
   , {0.1799999999999999, {1, 1, 1, 1, 1, 2, 2}, 1.285714285714284, 
 
   0.4879500364742666, {1}, 1., 0, {1, 1, 1, 3, 2, 1, 3}, 1.714285714285713, 
 
   0.951189731211342}}
;[o]
{{0, {11, 7, 1, 2, 1, 2, 1, 4, 2, 9, 5, 2, 12, 3, 2}, 4.266666666666665, 3.731462116089284, {}, 0, 0, {}, 0, 0}, {0.03, {2, 3, 2, 4, 2, 5, 2, 1, 2, 1, 1, 5, 1}, 2.384615384615384, 1.445594545418454, {}, 0, 0, {1, 1}, 1., 0}, {0.06, {1, 5, 1, 1, 1, 6, 3, 1, 1, 2, 6}, 2.545454545454544, 2.114882330704777, {3}, 3., 0, {3, 2, 4}, 3., 1.}, {0.09, {4, 2, 2, 1, 2, 3, 7, 2, 1, 5, 5}, 3.09090909090909, 1.921173883569388, {2}, 2., 0, {1, 1, 1}, 1., 0}, {0.1199999999999999, {1, 2, 2, 1, 1, 1, 5, 7}, 2.5, 2.267786838055362, {3}, 3., 0, {2, 2, 1, 1, 1, 1}, 1.333333333333333, 0.516397779494322}, {0.15, {1, 1, 1, 4, 1, 2, 1}, 1.57142857142857, 1.133893419027681, {10, 4, 1}, 5., 4.582575694955838, {1, 1, 3, 1, 2}, 1.6, 0.894427190999916}, {0.1799999999999999, {1, 1, 1, 1, 1, 2, 2}, 1.285714285714284, 0.4879500364742667, {1}, 1., 0, {1, 1, 1, 3, 2, 1, 3}, 1.714285714285713, 0.951189731211342}}
:[font = subsubsection; inactive; preserveAspect; startGroup]
Code for CollectBugStatistics in here ...
:[font = input; initialization; preserveAspect; endGroup]
*)
CollectBugStatistics[inState_, correctOutState_, circuit_, nRuns_, 
   {stdDevErrorMin_, stdDevErrorMax_, stdDevErrorIncr_}, file_] := 
  Table[SimulateBugsInFeynmanQC[inState, correctOutState, circuit, nRuns, 
    stdDevError, file, Report -> False], 
   {stdDevError, stdDevErrorMin, stdDevErrorMax, stdDevErrorIncr}]

CollectBugStatistics::usage = "CollectBugStatistics[inputState, \
correctOutputState, circuit, n, {stdDevErrorMin, stdDevErrorMax, \
stdDevErrorIncr},file] performs n runs of the computation implied by the \
given circuit for each value of the error in the range stdDevErrorMin to \
stdDevErrorMax in steps of size stdDevErrorIncr. The output is the set of \
summary statistics for each data point as returned by a call to \
SimulateBugsInFeynmanQC for a specific value of stdDevError."; 
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = text; inactive; preserveAspect]
To plot the fraction of runs that ended, at time t=2, with an uncorrupted cursor and the correct answer to the computation, use PlotFractionCorrectAtTimeTVersusStdDevError[bugStatistics, 2]
:[font = input; preserveAspect; endGroup]
PlotFractionCorrectAtTimeTVersusStdDevError[bugStatistics, 2]; 
:[font = subsubsection; inactive; preserveAspect; startGroup]
Code for PlotFractionCorrectAtTimeTVersusStdDevError in here ...
:[font = input; initialization; preserveAspect; endGroup; endGroup; endGroup]
*)
PlotFractionCorrectAtTimeTVersusStdDevError[bugStats_, t_] := 
  Module[{data}, data = 
     ({ExtractStdDevError[#1], FractionCorrectAtTimeT[#1, t], 
         StdErrorOfMeanFractionCorrectAtTimeT[#1, t]} & ) /@ bugStats; 
    ErrorListPlot[data, PlotRange -> {0, 1.1}, Frame -> True, 
     FrameLabel -> {"stdDevError", "fraction correct", "", ""}]]

ExtractStdDevError[{stdDevError_, _, _, _, _, _, _, _, _, _}] := 
	stdDevError

FractionCorrectAtTimeT[{_, r_, _, _, w_, _, _, u_, _, _}, t_] := 
  Module[{numerator}, numerator = Length[Select[r, #1 == t & ]]; 
    If[numerator > 0, N[numerator/Length[Select[Join[r, w, u], #1 == t & ]]], 
     0]]

StdErrorOfMeanFractionCorrectAtTimeT[{sDE_, 
                                      r_, mr_, sr_, 
                                      w_, mw_, sw_, 
                                      u_, mu_, su_}, t_] := 
  Module[{f}, f = 
     FractionCorrectAtTimeT[{sDE, r, mr, sr, w, mw, sw, u, mu, su}, t]; 
    If[f > 0, Sqrt[((1 - f)*f)/Length[Select[Join[r, w, u], #1 == t & ]]], 0]]
(*
^*)
